/*
	Visual Plug-in API
	1999, @soft
	
	Description:	The plugin API for MacAMP visual plugins (windowed
					and full-screen).
	Version:		1.2
	Released:		7/11/99
	Compatibility:	MacAMP 1.0
	Version history:
	
	 Date	Who		Changes
	------+------+------------------------------------------------------
	071199	SKA		Changed a few callbacks, commented the file a little
					better.
	070899	SKA		Added two new hooks for plugins. 
					All visual plugins has to be recompiled.
	061999	SKA		Added GetSoundBuffer hook for direct access to sound
					data
	050299	SKA		Initial release
	043099	SKA		Started the work

	SKA = Slava Karpenko
*/

#pragma once

#include <Errors.h>
#include <Files.h>
#include <MacTypes.h>

#ifdef __cplusplus
extern "C" {
#endif


/************************************************************
					ENUMS AND CONSTANTS
************************************************************/
#define VP_API_VERSION				0x0120
#define plugVisual					'VIS!'

// Flags for VisInfoBlock.flags
enum {
	vpFlagStartOnLaunch			= 0x0001,		// The plugin will be started on application launch time
	vpFlagHideInMenu			= 0x0002,		// The plugin will not be shown in menu
	
	vpFlagStartOnLaunchMask		= (1L << 0),
	vpFlagHideInMenuMask		= (1L << 1)
};

// useful macros to quickly set headers/footers of VPInfoBlock
#define VP_INFOBLOCK_HEADER(a,b) VP_API_VERSION, plugVisual, a, b, nil
#define VP_INFOBLOCK_HEADER_WITH_FLAGS(a,b,c) VP_API_VERSION, plugVisual, a, b, c
#define VP_INFOBLOCK_FOOTER	nil,nil,nil,nil,nil

/************************************************************
					ERRORS & RETURN VALUES
************************************************************/
enum {
	// Plugin return codes
	errVisNoErr			= noErr,
	errVisTerminate	= 666,			// terminate the plugin
	errVisNoMemory,					// terminate the plugin and display no memory alert
	
	errVisCustom					// MacAMP will call VisError function and terminate your
									//  plugin if you request to.
};

/************************************************************
					FUNCTION DEFINITIONS
************************************************************/
// Called when the plugin is initialized. You get the FSSpec to the plugin file
// (for opening resfork if you need to). You shall return WindowPtr to your plugin (if you create any).
extern OSStatus	VisInitialize(FSSpecPtr inPlugin, WindowPtr* outWindow, UInt32* ioRefcon);

// Called on termination. Plugin should dispose all buffers/structures it had and prepare
// the mind for ethernity. Disposing window is your plugin's responsibility as well.
extern OSStatus	VisTerminate(WindowPtr inWindow, UInt32* ioRefcon);

// Called whenever MacAMP has some free time. If you're a full-screen plugin, you get this event more often.
extern OSStatus	VisIdle(WindowPtr inWindow, UInt32* ioRefcon);

// Display About box or do some about graphic in the plugin window.
extern OSStatus	VisAbout(WindowPtr inWindow, UInt32* ioRefcon);

// Draw whatever you want to. Called periodically to ensure steady effect flow.
extern OSStatus VisDraw(WindowPtr inWindow, UInt32* ioRefcon);

// This function is called when user clicks in window that is controlled by your plugin.
// You can do what you want with that click.
extern OSStatus VisClick(WindowPtr inWindow, Point inClick, UInt32* ioRefcon);

// This function is called when user presses a key. Note that MacAMP gets this key as well
// as all other visual plugins.
extern OSStatus VisKeyDown(WindowPtr inWindow, EventRecord* inEvent, UInt32* ioRefcon);

// This function is called when window-specific event is posted for your visual plugin.
// If you have this set, it can override VisDraw function, and it is called when MacAMP
// wants your plugin to redraw.
extern OSStatus VisMacEvent(WindowPtr inWindow, EventRecord* inEvent, UInt32* ioRefcon);

// This function called if you return errVisCustom. You should copy the error string
// and errNum (optionally) and return whether the error is fatal or not. If this function
// returns true, MacAMP terminates your plugin, otherwise continues execution.
extern Boolean VisError(StringPtr outErrString, OSStatus* outErrNum);

// If you define this function in your gPlugInfo, MacAMP will add a corresponding item into
// 'Settings' submenu of plugins menu and will call this function when user chooses it.
// You can display settings dialog, and do whatever you want.
extern OSStatus VisSettings(WindowPtr inWindow, UInt32* ioRefcon);

// These functions are called when track in MacAMP is started or stopped. You can clear draw area
// etc here. Note that you can also use ma->GetStatus callback to get the current status.
extern OSStatus VisTrackBegin(WindowPtr inWindow, UInt32* ioRefcon);
extern OSStatus VisTrackEnd(WindowPtr inWindow, UInt32* ioRefcon);

/************************************************************
					TYPEDEFS AND STRUCTS
************************************************************/
// Plugin callbacks
typedef OSStatus	(*visInitProcPtr)(FSSpecPtr, WindowPtr*, UInt32*);
typedef OSStatus	(*visGenericProcPtr)(WindowPtr, UInt32*);
typedef OSStatus	(*visClickProcPtr)(WindowPtr, Point, UInt32*);
typedef OSStatus	(*visEventProcPtr)(WindowPtr, EventRecord*, UInt32*);
typedef Boolean 	(*visErrorProcPtr)(StringPtr, OSStatus*);

// MacAMP Callbacks
//  You can call these anytime you want.

typedef void		(*maVisGetValuesArray)(UInt8** outArray, UInt16* outArraySize);
// Get an array of pre-computed values for frequences.
// Every value represents one frequency, and can be in range #0-232.
// Input:
//		none
//	Output:
//		outArray		= pointer to values array
//		outArraySize	= size of the array
//	Returns:
//		nothing

typedef void		(*maVisGetFFTArray)(double** outFFT, UInt16* outFFTSize);
// Get FFT array. First half of the array are real parts, second - imaginary.
// Input:
//		none
//	Output:
//		outFFT			= pointer to raw fft array (or nil, if not available)
//		outBufferSize	= size of the array
//	Returns:
//		nothing

typedef void		(*maVisGetSoundBuffer)(Ptr* outSoundBuffer, UInt16* outBufferSize);
// Get current sound buffer.
// Input:
//		none
//	Output:
//		outSoundBuffer	= pointer to sound buffer being played (or nil, if no track is playing)
//		outBufferSize	= size of the buffer, in bytes
//	Returns:
//		nothing

typedef void		(*maVisGetStatus)(UInt32* outTimer, UInt32* outStatus);
// Get status of thge player (timer, states of repeat, random, sleep etc).
// Input:
//		none
//	Output:
//		outTimer			= timer, in seconds
//		outStatus			= flags field containing status* constants.
//								(for example, to check if Random mode is set, check for (outStatus & statusRandom))
//	Returns:
//		nothing

// Status codes
enum {
	statusRandom	= (1L << 0),
	statusSleep		= (1L << 1),
	statusRepeat	= (1L << 2),
	statusRepeat1	= (1L << 3),
	statusPlaying 	= (1L << 4),
	statusPaused	= (1L << 5),
	statusStopped 	= (1L << 6)
};

typedef OSStatus	(*maVisGetTrackLocationProcPtr)(OSType* outType, Ptr* outData);
// Get current track location.
// Input:
//		none
//	Output:
//		outType			= location type, either locationFile, locationURL or locationNone
//		outData			= location data (depends on outType, see below)
//	Returns:
//		noErr			= lyrics information fetched.
//		resNotFound		= no track is playing at the moment.

// Location types
enum {
	locationFile	= 'file',		// outData = FSSpecPtr
	locationURL		= 'url ',		// outData = char* (C string)
	locationNone	= 'none'		// outData = void, does not contains valid data
};

typedef OSStatus	(*maVisGetID3InformationProcPtr)(StringPtr outArtist, StringPtr outTitle, StringPtr outAlbum, StringPtr outYear, Ptr* outLyrics);
// Get current track ID3 information.
// Input:
//		none
//	Output:
//		outArtist		= artist
//		outTitle		= title
//		outAlbum		= album (if available, or nil)
//		outYear			= year (if available, or nil)
//		outLyrics		= lyrics (if available, or nil)
//	Returns:
//		noErr			= lyrics information fetched.
//		resNotFound		= no track is playing at the moment.

typedef QDGlobalsPtr (*maVisGetQDPtrProcPtr)(void);
// Return the pointer to QD globals (qd->...).
// Input:
//		none
//	Output:
//		none
//	Returns:
//		pointer to qd globals

typedef void (*maVisEnterFullScreen)(OSType inAuthorID, OSType inPluginID);
// Notify MacAMP of your plugin entering full-screen mode.
// After this call MacAMP hides all its windows, and suspends processing other visual plugins
// to ensure maximum CPU time available for your plugin.
// Input:
//		inAuthorID		= author ID as defined in VisInfoBlock structure
//		inPlugID		= plugin ID as defined in VisInfoBlock structure
//	Output:
//		none
//	Returns:
//		nothing
//
// Note: if you pass author/plugin IDs different than defined in your VisInfoBlock, that could cause
// abnormal MacAMP behavior.

typedef void (*maVisExitFullScreen)(void);
// Notify MacAMP of your plugin exiting full-screen mode.
// All previously hidden windows will be shown again, and other visual plugins will get events.
// Input:
//		none
//	Output:
//		none
//	Returns:
//		nothing

typedef OSStatus (*maVisSavePrefs)(OSType inAuthorID, OSType inPluginID, Ptr inBuffer, UInt16 inSize);
// Save plugin-specific preferences in MacAMP Data file located in System Folder.
// Input:
//		inAuthorID 		= authorID of plugin
//		inPluginID 		= pluginID of plugin
//		inBuffer		= pointer to buffer containing preferences chunk to be saved
//		inSize			= size of buffer to be saved
//	Output:
//		none
//	Returns:
//		noErr			= preferences saved successfully.
//		memFullErr		= an unknown, probably memory-related error has occurred.

typedef OSStatus (*maVisReadPrefs)(OSType inAuthorID, OSType inPluginID, Ptr inBuffer, UInt16* ioSize);
// Read plugin-specific preferences you have saved before with SavePrefs hook.
// Input:
//		inAuthorID 		= authorID of plugin
//		inPluginID 		= pluginID of plugin
//		inBuffer		= pointer to pre-allocated buffer to store preferences
//		ioSize			= size of allocated buffer
//	Output:
//		ioSize			= actual size of preferences
//	Returns:
//		noErr			= preferences read successfully.
//		resNotFound		= unable to find such preferences chunk
//		memFullErr		= size of preferences chunk exceeds size of allocated buffer
//							(ioSize contains size of preferences chunk)

// Various structures
#pragma options align=power

// This is the structure for MacAMP callbacks.
typedef struct {
	maVisGetValuesArray				GetValues;
	maVisGetFFTArray				GetFFT;
	maVisGetStatus					GetStatus;
	maVisGetTrackLocationProcPtr	GetTrackLocation;
	maVisGetID3InformationProcPtr	GetID3Information;
	maVisGetQDPtrProcPtr			GetQDGlobals;
	
	// Full-screen mode
	maVisEnterFullScreen			EnterFullScreen;
	maVisExitFullScreen				ExitFullScreen;
	
	// Storing and reading prefs
	maVisReadPrefs					ReadPrefs;
	maVisSavePrefs					SavePrefs;

	// New for 1.01
	maVisGetSoundBuffer				GetSoundBuffer;
} VPCallbacks, *VPCallbacksPtr;

// Main structure for the plugin. Main entry point for the PEF fragment should point to
// a global variable of this type.
typedef struct {
	UInt16	api;					// API used. Should always contain VP_API_VERSION
	OSType	type;					// Plugin type. Should be plugVisual.
	
	OSType	authorID;				// Author ID should be registered with @soft to prevent
									//  incompatibilities. mailto:devsupport@at-soft.net or
									//  mailto:devsupport@macamp.com
	OSType	pluginID;				// Plugin ID can be anything the author wishes. It is
									//  used with author id to store plugin preferences etc
	
	UInt32	flags;					// Plugin flags
	
	Str255	name;					// Plugin name (displayed in Plugins menu)

	visInitProcPtr			initProc;
	visGenericProcPtr		terminateProc;
	visGenericProcPtr		idleProc;		// set to nil if you don't want to be called at idle time
	visGenericProcPtr		aboutProc;

	visGenericProcPtr		drawProc;
	visClickProcPtr			clickProc;
	
	visEventProcPtr			keyDownProc;	// set to nil if you don't want keydowns.
	visEventProcPtr			macEventProc;	// set to nil if you don't want to handle update/click events manually.
	
	visErrorProcPtr			errorProc;
	visGenericProcPtr		settingsProc;	// set to nil if you don't want to be displayed in settings submenu
	visGenericProcPtr		trackBeginProc;
	visGenericProcPtr		trackEndProc;
	
	// Set these to nil. They are reserved for future expansion.
	visGenericProcPtr		reservedProc1;
	visGenericProcPtr		reservedProc2;
	visGenericProcPtr		reservedProc3;
	visGenericProcPtr		reservedProc4;
		
	// Special APIs (Set to nil, MacAMP will set it to the right value)
	VPCallbacksPtr				ma;
} VPInfoBlock, *VPInfoPtr;
#pragma options align=reset

#ifdef __cplusplus
}
#endif
